% Steering A Linear Array Example
%
% This example demonstrates how to use k-Wave to steer a tone burst from a
% linear array transducer. It builds on the Simulating Transducer Field
% Patterns Example.
%
% author: Bradley Treeby
% date: 10th December 2009
% last update: 7th February 2011
%  
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010, 2011 Bradley Treeby and Ben Cox

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>. 

clear all;

% =========================================================================
% SIMULATION
% =========================================================================

% create the computational grid
Nz = 128;           % number of pixels in the x (column) direction
Nx = Nz;            % number of pixels in the z (row) direction
dz = 50e-3/Nz;    	% pixel width [m]
dx = dz;            % pixel height [m]
kgrid = makeGrid(Nx, dx, Nz, dz);

% define the properties of the propagation medium
medium.sound_speed = 1500;  % [m/s]
medium.alpha_power = 1.5;   % [dB/(MHz^y cm)]
medium.alpha_coeff = 0.75;  % [dB/(MHz^y cm)]

% create the time array
[kgrid.t_array dt] = makeTime(kgrid, medium.sound_speed, 0.3);
kgrid.t_array = 0:dt:500*dt;

% define a linear element transducer with an odd number of elements
num_elements = 21;
z_offset = 25;
source.p_mask = zeros(Nz, Nx);
start_index = Nz/2 - round(num_elements/2) + 1;
source.p_mask(z_offset, start_index:start_index + num_elements - 1) = 1;

% define the properties of the tone burst using a geometrically steered
% transmit pulse offset
steering_angle = 30;
element_spacing = dz;
element_index = -(num_elements - 1)/2:(num_elements - 1)/2;
tone_burst_offset = 40 + element_spacing*element_index*sin(steering_angle*pi/180)/(medium.sound_speed*dt);
tone_burst_freq = 1e6;
tone_burst_cycles = 8;
tone_burst_signal_length = length(kgrid.t_array);
sampling_freq = 1/dt;
source.p = toneBurst(sampling_freq, tone_burst_freq, tone_burst_signal_length, tone_burst_offset, tone_burst_cycles);

% assign the input options
input_args = {'DisplayMask', source.p_mask};

% run the simulation
kspaceFirstOrder2D(kgrid, medium, source, [], input_args{:});

% =========================================================================
% VISUALISATION
% =========================================================================

% plot the input time series
figure;
[t_sc scale prefix] = scaleSI(max(kgrid.t_array(:)));
plot(kgrid.t_array*scale, source.p + repmat(2*(1:num_elements).', 1, length(kgrid.t_array)), 'k-');
xlabel(['Time [' prefix 's]']);
ylabel('Input Signals');
axis tight;
set(gca, 'XLim', [0 20]);